#!/usr/bin/env bash

set -Eeuo pipefail
trap cleanup SIGINT SIGTERM ERR EXIT

script_dir=$(cd "$(dirname "${BASH_SOURCE[0]}")" &>/dev/null && pwd -P)

usage() {
  cat <<EOF
Usage: $(basename "${BASH_SOURCE[0]}") [-h] [-v] [-d] -z timezone -t timestamp --dhcp -i ipaddress -n netmask -g gateway

Get or set device settings.

Available options:

-h, --help      Print this help and exit
-v, --verbose   Print script debug info
-d, --dump      Print system information to stdout
-z, --timezone  Set time zone (e.g. Europe/Zurich or UTC)
-t, --timestamp Set system time (UTC) unix timestamp
--dhcp          Set DHCP mode of eth0
-i, --ipaddress Set IP address of eth0 (don't use with --dhcp)
-n, --netmask   Set the netmask of eth0 (don't use with --dhcp)
-g, --gateway   Set standard gateway on eth0 (don't use with --dhcp)
-o, --hostname  Set the hostname
EOF
  exit
}

cleanup() {
  trap - SIGINT SIGTERM ERR EXIT
  # script cleanup here
}

setup_colors() {
  if [[ -t 2 ]] && [[ -z "${NO_COLOR-}" ]] && [[ "${TERM-}" != "dumb" ]]; then
    NOFORMAT='\033[0m' RED='\033[0;31m' GREEN='\033[0;32m' ORANGE='\033[0;33m' BLUE='\033[0;34m' PURPLE='\033[0;35m' CYAN='\033[0;36m' YELLOW='\033[1;33m'
  else
    NOFORMAT='' RED='' GREEN='' ORANGE='' BLUE='' PURPLE='' CYAN='' YELLOW=''
  fi
}

msg() {
  echo >&2 -e "${1-}"
}

die() {
  local msg=$1
  local code=${2-1} # default exit status 1
  msg "$msg"
  exit "$code"
}

parse_params() {
  # default values of variables set from params
  dump=0
  hostname=''
  timezone=''
  timestamp=''
  usedhcp=0
  ipaddress=''
  netmask=''
  gateway=''

  while :; do
    case "${1-}" in
    -h | --help) usage ;;
    -v | --verbose) set -x ;;
    --no-color) NO_COLOR=1 ;;
    -d | --dump) dump=1 ;;
    -o | --hostname)
      hostname="${2-}"
      shift
      ;;
    -z | --timezone)
      timezone="${2-}"
      shift
      ;;
    -t | --timestamp)
      timestamp="${2-}"
      shift
      ;;
    --dhcp) usedhcp=1 ;;
    -i | --ipaddress)
      ipaddress="${2-}"
      shift
      ;;
    -n | --netmask)
      netmask="${2-}"
      shift
      ;;
    -g | --gateway)
      gateway="${2-}"
      shift
      ;;
    -?*) die "Unknown option: $1" ;;
    *) break ;;
    esac
    shift
  done

  args=("$@")
  return 0
}

parse_params "$@"
setup_colors

eth0service=$(connmanctl services | (grep 'ethernet_.*_cable' || [[ $? == 1 ]]) | awk '{print $NF}')

if [[ ${dump} -eq 1 ]]; then
  unixepoch=$(date +%s)
  echo "timezone: $(timedatectl show -p Timezone --value)"
  echo "timestamp: ${unixepoch}"

  echo "hostname: $(cat /etc/hostname)"

  eth0dhcp=
  eth0ipaddress=
  eth0netmask=
  eth0gateway=
  if [[ ! -z ${eth0service} ]]; then
    eth0properties=$(connmanctl services ${eth0service} | (grep 'IPv4 =' || [[ $? == 1 ]]) | cut -d '[' -f 2 | cut -d ']' -f 1 )
    if [[ ! -z ${eth0properties} ]]; then
      eth0method=$(echo ${eth0properties} | cut -d ',' -f 1 | cut -d '=' -f 2)
      eth0dhcp=$(if [[ "${eth0method}" == "dhcp" ]]; then echo 1; else echo 0; fi)
      eth0ipaddress=$(echo ${eth0properties} | cut -d ',' -f 2 | cut -d '=' -f 2)
      eth0netmask=$(echo ${eth0properties} | cut -d ',' -f 3 | cut -d '=' -f 2)
      eth0gateway=$(echo ${eth0properties} | cut -d ',' -f 4 | cut -d '=' -f 2)
    fi
  fi
  echo "dhcp: ${eth0dhcp}"
  echo "ipaddress: ${eth0ipaddress}"
  echo "netmask: ${eth0netmask}"
  echo "gateway: ${eth0gateway}"
  echo "osversion: $(cat /REVISION)"
  echo "osbuild: $(cat /REVBUILD)"
  echo "swversion: $(cat ${script_dir}/../REVISION)"
  echo "swbuild: $(cat ${script_dir}/../REVBUILD)"
  exit 0;
fi

if [[ -n "${hostname-}" ]] && \
   [[ "${hostname-}" != "$(hostnamectl --static)" ]]; then
   echo set hostname to ${hostname}
   hostnamectl set-hostname ${hostname}
fi

if [[ -n "${timezone-}" ]]; then
  echo set time zone to ${timezone}
  timedatectl set-timezone ${timezone}
fi

if [[ -n "${timestamp-}" ]]; then
  echo set UTC timetamp to ${timestamp}
  timedatectl set-time @${timestamp}
fi

if [[ ${usedhcp} -eq 1 ]]; then
  [[ -z "${eth0service-}" ]] && die "Ethernet not connected"
  echo "set eth0 IPv4 mode to DHCP"
  connmanctl config ${eth0service} --ipv4 dhcp --nameservers
else
  if [[ -n "${ipaddress-}" ]] || \
     [[ -n "${netmask-}"   ]] || \
     [[ -n "${gateway-}"   ]]; then

    [[ -z "${eth0service-}" ]] && die "Ethernet not connected"

    # assert all parameters are present
    [[ -z "${ipaddress-}" ]] && die "Missing required parameter: ipaddress"
    [[ -z "${netmask-}"   ]] && die "Missing required parameter: netmask"
    [[ -z "${gateway-}"   ]] && die "Missing required parameter: gateway"
    echo "set eth0 IPv4 mode to manual (${ipaddress} ${netmask} ${gateway})"
    connmanctl config ${eth0service} --ipv4 manual ${ipaddress} ${netmask} ${gateway}
  fi
fi
