#!/usr/bin/env bash

set -Eeuo pipefail
trap cleanup SIGINT SIGTERM ERR EXIT

script_dir=$(cd "$(dirname "${BASH_SOURCE[0]}")" &>/dev/null && pwd -P)

usage() {
  cat <<EOF
Usage: $(basename "${BASH_SOURCE[0]}") [-h] [--off | --debug | --trace] [--commlog | --nocommlog]

Configure the different logging levels.

Available options:

-h, --help      Print this help and exit
--off           disable detailed logging
--debug         enable debug logging
--trace         enable trace logging
--commlog       enable communication logging
--nocommlog     disable communication logging
EOF
  exit
}

cleanup() {
  trap - SIGINT SIGTERM ERR EXIT
  # script cleanup here
}

setup_colors() {
  if [[ -t 2 ]] && [[ -z "${NO_COLOR-}" ]] && [[ "${TERM-}" != "dumb" ]]; then
    NOFORMAT='\033[0m' RED='\033[0;31m' GREEN='\033[0;32m' ORANGE='\033[0;33m' BLUE='\033[0;34m' PURPLE='\033[0;35m' CYAN='\033[0;36m' YELLOW='\033[1;33m'
  else
    NOFORMAT='' RED='' GREEN='' ORANGE='' BLUE='' PURPLE='' CYAN='' YELLOW=''
  fi
}

msg() {
  echo >&2 -e "${1-}"
}

die() {
  local msg=$1
  local code=${2-1} # default exit status 1
  msg "$msg"
  exit "$code"
}

parse_params() {
  # default values of variables set from params
  traceLevel=''
  commLevel=''

  while :; do
    case "${1-}" in
    -h | --help) usage ;;
    --no-color) NO_COLOR=1 ;;
    --off) traceLevel='off' ;;
    --debug) traceLevel='debug' ;;
    --trace) traceLevel='trace' ;;
    --commlog) commLevel='trace' ;;
    --nocommlog) commLevel='off' ;;
    -?*) die "Unknown option: $1" ;;
    *) break ;;
    esac
    shift
  done

  args=("$@")
  return 0
}

parse_params "$@"
setup_colors

if [[ -n "${traceLevel-}" ]]; then
   echo set trace log level to ${traceLevel}
   sed -i "s|name=\"traceLevel\" value=\".*\"|name=\"traceLevel\" value=\"${traceLevel}\"|" ${script_dir}/../nlog.config
fi

if [[ -n "${commLevel-}" ]]; then
   echo set communication log level to ${commLevel}
   sed -i "s|name=\"commLevel\" value=\".*\"|name=\"commLevel\" value=\"${commLevel}\"|" ${script_dir}/../nlog.config
fi

if [[ -n "${traceLevel-}" ]] || [[ -n "${commLevel-}" ]]; then
    touch ${script_dir}/../nlog.config
fi
