#!/usr/bin/env bash

set -Eeuo pipefail
trap cleanup SIGINT SIGTERM ERR EXIT

script_dir=$(cd "$(dirname "${BASH_SOURCE[0]}")" &>/dev/null && pwd -P)

usage() {
  cat <<EOF
Usage: $(basename "${BASH_SOURCE[0]}") [-h] <output zip>

Collect log files and packed them in a zip file.

Available options:

-h, --help      Print this help and exit
EOF
  exit
}

cleanup() {
  trap - SIGINT SIGTERM ERR EXIT
  # script cleanup here
}

setup_colors() {
  if [[ -t 2 ]] && [[ -z "${NO_COLOR-}" ]] && [[ "${TERM-}" != "dumb" ]]; then
    NOFORMAT='\033[0m' RED='\033[0;31m' GREEN='\033[0;32m' ORANGE='\033[0;33m' BLUE='\033[0;34m' PURPLE='\033[0;35m' CYAN='\033[0;36m' YELLOW='\033[1;33m'
  else
    NOFORMAT='' RED='' GREEN='' ORANGE='' BLUE='' PURPLE='' CYAN='' YELLOW=''
  fi
}

msg() {
  echo >&2 -e "${1-}"
}

die() {
  local msg=$1
  local code=${2-1} # default exit status 1
  msg "$msg"
  exit "$code"
}

parse_params() {
  # default values of variables set from params
  outputzipfile=''

  if [[ -z "${@//[[:blank:]]}" ]]; then
    # no arguments available (or only whitespaces)
    usage
  fi;

  while :; do
    case "${1-}" in
    -h | --help) usage ;;
    --no-color) NO_COLOR=1 ;;
    -?*) die "Unknown option: $1" ;;
    *)
      outputzipfile=${1-}
      shift
      break;;
    esac
    shift
  done

  args=("$@")
  
  return 0
}

parse_params "$@"
setup_colors

if [[ -z "${outputzipfile-}" ]]; then
    die "output zip file name is missing."
fi;

if [[ -e "${outputzipfile}" ]]; then
    msg "remove existing file: ${outputzipfile}"
    rm "${outputzipfile}"
fi;

if [[ -n "${outputzipfile}" ]]; then
    pushd /var/www/
    # create a file with version information
    if [[ -e "/usr/bin/versions" ]]; then
        /usr/bin/versions > mini-atos-log/versions.txt
    else
        echo "osversion: $(cat /REVISION) - $(cat /REVBUILD)" > mini-atos-log/versions.txt
        echo "swversion: $(cat ${script_dir}/../REVISION) - $(cat ${script_dir}/../REVBUILD)" >> mini-atos-log/versions.txt
    fi
    # zip mini-atos-log directory
    zip -r ${outputzipfile} mini-atos-log/*
    popd

    pushd /data/
    if [[ -e "/data/log/rauc-status.log" ]]; then
        # log rauc current status
        echo "Rauc status at `date --rfc-3339=seconds`" > "log/rauc-status-tmp.txt"
        rauc status --detailed >> "/data/log/rauc-status-tmp.txt"
    fi
    # zip /data/log directory
    zip -r ${outputzipfile} log/*
    popd

    if [[ -e "/opt/firmware/rauc" ]]; then
        pushd /opt/firmware
        # log rauc current status
        rauc status --detailed --output-format=json-pretty >> "rauc/rauc-status-tmp.json"
        # zip /opt/firmware/rauc directory
        zip -r ${outputzipfile} rauc/*
        popd
    fi
fi;
